//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.2
import QtGraphicalEffects 1.0

import wesual.Controls 1.0
import wesual.data.Core 1.0

FocusScope {
    id : searchBar

    property list<StringFilter> filters

    property alias  inputHasActiveFocus     : textField.activeFocus
    property alias  placeholderText         : textField.placeholderText
    property bool   buttonOnlyWhenInactive  : false
    property bool   stayActive              : false
    property int    fullSizeWidth

    signal filterReset()

    function forceFocus() {
        textField.forceActiveFocus();
    }

    function minimize() {
        textField.focus = false;
        p_.dontResize = false;
        for(var idx = 0; idx < filters.length; idx++){
            filters[idx].propertyFilter = "";
        }
    }

    QtObject {
        id : p_
        property bool dontResize : false
    }
    enabled : true
    focus : true
    activeFocusOnTab : true

    onActiveFocusChanged : {
        if (activeFocus)
            forceFocus();
    }

    states : [
        State {
            name : "button"
            when : buttonOnlyWhenInactive && (!p_.dontResize &&
                                              !textField.activeFocus
                   && (textField.text.length === 0))

            PropertyChanges {
               target : searchBar
               width : iconContainer.width
            }
            PropertyChanges {
                target : textField
                width : 0
                opacity : 0
                text : ""
            }
            PropertyChanges {
                target : topcontainer
                opacity : 0
            }
            onCompleted : bg.color = UI.color(UI.PrimaryBase)
        },
        State {
            name : "fullsize"
            when : buttonOnlyWhenInactive && (p_.dontResize
                                              || (textField.text.length > 0)
                                              || (textField.activeFocus))

            PropertyChanges {
               target : searchBar
               width : fullSizeWidth
            }
            PropertyChanges {
               target : bg
               opacity : 1
            }
            PropertyChanges {
                target : topcontainer
                opacity : 1
            }
            onCompleted : bg.color = Qt.binding(function(){
                return textField.activeFocus || textField.text.length > 0 ||
                        searchBar.state === "fullsize" ?
                        UI.color(UI.PrimaryPress) : UI.color(UI.PrimaryBase)
            });
        }
    ]
    transitions : Transition {
        NumberAnimation {
            properties : "width, opacity"
            duration : 300
        }
    }

    width :  100
    implicitHeight : 30

    Rectangle {
        id : bg
        anchors.fill : parent
        opacity : buttonOnlyWhenInactive ? 0 : 1
        color : textField.activeFocus || textField.text.length > 0 ?
                    UI.color(UI.PrimaryPress) : UI.color(UI.PrimaryBase)
        Behavior on color {
            ColorAnimation { duration : 50 }
        }
    }

    Item {
        id : contentWrapper
        anchors.fill : parent

        Rectangle {
            id : topcontainer
            height : 26
            width : parent.width - 4
            anchors.centerIn : parent
            border {
                width : (textField.width > 1) ? 1 : 0
                color : textField.text.length > 0 ? UI.color(UI.PrimaryPress) :
                                                    textField.border.color
            }

            Behavior on border.color {
                ColorAnimation { duration : 0 }
            }

            TextField {
                id : textField

                Timer {
                    id : timer;
                    interval : 500;
                    running : false;
                    repeat : false;

                    // when input delay elapsed start filtering
                    onTriggered : {
                        if(textField.activeFocus){
                            for(var idx = 0; idx < filters.length; idx++){
                                filters[idx].propertyFilter = textField.text;
                            }
                        }
                    }
                }

                backgroundColor : topcontainer.color
                border.width : 0
                anchors {
                    fill : parent
                    margins : topcontainer.border.width
                    leftMargin : 2
                    rightMargin: iconContainer.width
                }
                placeholderText : qsTrId("0918382c8d4b362e")
                textColor : text.length ?
                            UI.color(UI.PrimaryTextColor) : searchBar.enabled ?
                                UI.color(UI.SecondaryTextColor) :
                                UI.color(UI.SearchBarDisabledText)

                // search input
                onTextChanged : {
                    if(textField.activeFocus)
                        timer.restart();
                }
                onFocusChanged : {
                    if (focus)
                        p_.dontResize = stayActive;
                    else if (!focus && text.length === 0) {
                        minimize();
                    }
                }
                onEditFinished : {
                    timer.stop();
                    for(var idx = 0; idx < filters.length; idx++){
                        filters[idx].propertyFilter = textField.text;
                    }
                }
            }
        } // topcontainer

        Rectangle {
            id : iconContainer
            anchors {
                right : contentWrapper.right
                rightMargin : 3
                verticalCenter : parent.verticalCenter
            }
            height : 24
            width : 29
            opacity : 1
            color : {
                if(iconArea.pressed)
                    return UI.color(UI.PrimaryPress);
                if(iconArea.containsMouse)
                    return UI.color(UI.PrimaryHover);
                return UI.color(UI.PrimaryBase);
            }

            Behavior on color {
                ColorAnimation { duration : 0 }
            }

            property int duration : 500

            ColorizedImage {
                id : icon2
                anchors.fill : parent
                source : "qrc:/controls/icons/search-delete.png"
                opacity : 0
                color : searchBar.enabled ? "#4B4B4B" : "#BCBCBC"
            }

            ColorizedImage {
                id : icon
                anchors.fill : parent
                source : "qrc:/controls/icons/search-loupe.png"
                color : searchBar.enabled ? "#464646" : "#BCBCBC"
                states : [
                    State {
                        name : "swap"
                        when : searchBar.state === "fullsize" ||
                               textField.text.length > 0
                    },
                    State {
                        name : "swapBack"
                        when : searchBar.state === "button" ||
                               textField.focus === false
                    }
                ]
                transitions : [
                    Transition {
                        to : "swap"
                        NumberAnimation {
                            target: icon
                            properties : "opacity, scale"
                            easing.type : Easing.InOutQuad
                            duration : iconContainer.duration
                            to : 0.1
                        }
                        NumberAnimation {
                            target: icon2
                            properties : "opacity, scale"
                            easing.type : Easing.InOutQuad
                            duration : iconContainer.duration
                            to : 1
                        }
                    },
                    Transition {
                        to : "swapBack"
                        NumberAnimation {
                            target: icon2
                            properties : "opacity, scale"
                            easing.type : Easing.InOutQuad
                            duration : iconContainer.duration
                            to : 0.1
                        }
                        NumberAnimation {
                            target: icon
                            properties : "opacity, scale"
                            easing.type : Easing.InOutQuad
                            duration : iconContainer.duration
                            to : 1
                        }

                    }
                ]
            }

            MouseArea {
                id : iconArea
                hoverEnabled : true
                anchors.fill : parent
                acceptedButtons : Qt.LeftButton
                cursorShape : Qt.PointingHandCursor
                onPressed : {
                    textField.text = "";
                    if (searchBar.state === "button") {
                        textField.forceActiveFocus();
                        if (stayActive)
                            p_.dontResize = true;
                    } else {
                        if (stayActive && p_.dontResize)
                            p_.dontResize = !p_.dontResize;

                        for(var idx = 0; idx < filters.length; idx++){
                            filters[idx].propertyFilter = "";
                        }
                        textField.focus = false;
                        filterReset();
                    }
                }
            } // mouseArea
        }
    }
}
